from collections import deque
import itertools
import numpy as np
class WeightedMovingAverageFilter:
    def __init__(self, weights, disabled=False, threshold=0.95):
        self.weights = weights
        self.values = deque(maxlen=len(weights))
        self.disabled = disabled
        self.threshold = threshold

    def stable(self):
        cv = np.std(self.values) /  np.mean(self.values)
        return cv >= self.threshold
    
    def update(self, new_value):
        if self.disabled:
            return new_value

        self.values.append(new_value)
        if len(self.values) < len(self.weights):
            return new_value
        else:
            if self.stable():
                avg_x = sum(x for x, y in self.values) / len(self.values)
                avg_y = sum(y for x, y in self.values) / len(self.values)
                return (avg_x, avg_y)
            return [sum(w * v[i] for w, v in zip(self.weights, reversed(self.values))) for i in range(2)]


class ExponentialMovingAverageFilter:
    def __init__(self, alpha):
        self.alpha = alpha
        self.value = None

    def update(self, new_value):
        if self.value is None:
            self.value = new_value
        else:
            self.value = [(1 - self.alpha) * old + self.alpha * new for old, new in zip(self.value, new_value)]
        return self.value

if __name__ == "__main__":
    filter = WeightedMovingAverageFilter(weights=[0.6, 0.3, 0.1])
    coordinates = [(1920,1080), (1910, 1000), (1900, 990), (1890, 980)]
    for coord in coordinates:
        smoothed_value = filter.update(coord)
        print(smoothed_value)